from datetime import datetime
from lib.ClienteLinux import obter_cliente_linux
from lib.obter_configuracoes import obter_configuracoes

def volume_existe(volume: str, cliente_linux) -> bool:
    comando = f"docker volume inspect {volume}"
    resultado = cliente_linux.executar_comando(comando, disparar_excecao_se_retorno_nao_zero=False)
    return resultado.codigo_retorno == 0

def volume_esta_vazio(volume: str, cliente_linux) -> bool:
    comando = f"docker run --rm -v {volume}:/volume busybox sh -c 'du -sb /volume | cut -f1'"

    resultado = cliente_linux.executar_comando(comando, disparar_excecao_se_retorno_nao_zero=False)
    
    try:
        tamanho_bytes = int(resultado.saida.strip())
        # Considera vazio se tiver até 4KB (4096 bytes), tamanho típico de pasta vazia
        return tamanho_bytes <= 4096
    except (ValueError, AttributeError):
        print(f"[Verificação] Erro ao interpretar o tamanho do volume '{volume}'.")
        return True  

def fazer_backup_volume_docker(volume: str, arquivo_configuracao: str):
    configuracoes = obter_configuracoes(arquivo_configuracao)
    usuario = configuracoes['acessos']['servidorLinux']['usuario']
    cliente_linux = obter_cliente_linux(arquivo_configuracao)
    
    if not volume_existe(volume, cliente_linux):
        print(f"[Backup] Volume '{volume}' não existe. Backup cancelado.")
        return
        
    if volume_esta_vazio(volume, cliente_linux):
        print(f"[Backup] Volume '{volume}' está vazio. Backup não será realizado.")
        return


    data = datetime.now().strftime('%Y-%m-%d_%H-%M-%S')
    nome_arquivo_backup = f"backup_{volume}_{data}.tar.gz"
    caminho_backup = f"/home/{usuario}/backup"

    comando = (
        f"mkdir -p {caminho_backup} && "
        f"docker run --rm "
        f"-v {volume}:/volume "
        f"-v {caminho_backup}:/backup "
        f"busybox tar czf /backup/{nome_arquivo_backup} -C /volume ."
    )

    cliente_linux.executar_comando(comando)
    print(f"[Backup] Volume '{volume}' salvo como {caminho_backup}/{nome_arquivo_backup}")


def restaurar_backup_volume_docker(volume: str, arquivo_configuracao: str):
    configuracoes = obter_configuracoes(arquivo_configuracao)
    usuario = configuracoes['acessos']['servidorLinux']['usuario']
    cliente_linux = obter_cliente_linux(arquivo_configuracao)
    
    if not volume_existe(volume, cliente_linux):
        print(f"[Restaurar] Volume '{volume}' não existe. Restauração cancelada.")
        return
    
    if not volume_esta_vazio(volume, cliente_linux):
        print(f"[Restaurar] Volume '{volume}' NÃO está vazio. Restauração não será feita.")
        return

    caminho_backup = f"/home/{usuario}/backup"
    encontrar_backup = f"ls -1t {caminho_backup}/backup_{volume}_*.tar.gz 2>/dev/null | head -n 1"

    resultado_backup = cliente_linux.executar_comando(encontrar_backup, disparar_excecao_se_retorno_nao_zero=False)
    caminho_arquivo = resultado_backup.saida.strip()

    if not caminho_arquivo:
        print(f"[Restaurar] Nenhum backup encontrado para o volume '{volume}'.")
        return

    nome_arquivo_backup = caminho_arquivo.split('/')[-1]

    comando = (
        f"docker run --rm "
        f"-v {volume}:/volume "
        f"-v {caminho_backup}:/backup "
        f"busybox sh -c 'cd /volume && tar xzf /backup/{nome_arquivo_backup}'"
    )

    cliente_linux.executar_comando(comando)
    print(f"[Restaurar] Backup restaurado no volume '{volume}' a partir de {caminho_backup}/{nome_arquivo_backup}")
